/*
 * $Header: /home/kline/devel/atom/RCS/msg.c,v 1.1 1996/09/04 04:56:19 kline Exp kline $
 * msg.c - routines to handle debug and error messages
 */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include "msg.h"
/* 
 * Default levels for masking out messages.
 */
MSG_CONST msg_severity_t msg_level[MSG_CATEGORIES] = {
    5, 5, 0, 0, 0,
};

/* The string prepended to each message. You may use date directives
 * like %T for time (see strftime(3)). Extension: You can use %s for
 * the severity level.
 */
static const char *msg_preface[MSG_CATEGORIES] = {
    "Trace\t",
    "Debug\t",
    "Warning [%Y-%m-%d %H:%M.%S] level %s\n\t",
    "Error [%Y-%m-%d %H:%M.%S] level %s\n\t",
    "Fatal [%Y-%m-%d %H:%M.%S] level %s\n\t",
};

#if defined(MSG_COUNT_CALLS)
/* How many messages to accept in a given class before calling
 * MSG_EXIT.
 * 0 means never exit.
 */
int msg_warnings[MSG_CATEGORIES] = {
    0, 0, 0, 0, 0,
};
#endif

msg_category_t msg_category;	/* global used for communication from macro */
msg_severity_t msg_severity;	/* global used for communication from macro */
static FILE *msg_file;
static int msg_parse_preface[MSG_CATEGORIES];	/* really Boolean */
static int msg_init_done = 0;

/* Prototypes
 */
static void parse_envir(void);
static char *parse_preface(const char *);
static void msg_fatal(char *);


void
msg_init()
{
    const char *env;
    register short int i;

    msg_init_done = 1;
    env = getenv("MSG_FILE");
    if (env && *env) {
        msg_file = fopen(env, "a");
        if (msg_file == NULL) {
            perror(env);
            msg_file = stderr;
        }
	setbuf(msg_file, NULL);
    } else
        msg_file = stderr;

    for (i = MSG_CATEGORIES; i--;)
	msg_parse_preface[i] = strchr(msg_preface[i], '%') ? 1 : 0;

    parse_envir();
}


static void
parse_envir()
{
#if defined(MSG_PARSE_ENVIR)
    register short int         i = 0;
    const char *env;

    env = getenv("MSG_LEVEL");
    /* Parse the environment variable:
     *     "5,,-,"
     * becomes
     *     [ 5, OLD VALUE, MSG_MAX_SEVERITY + 1, OLD VALUE, OLD VALUE ]
     */
    if (env && *env) {
        enum { fresh, number, junk } state = fresh;
        register short int                          value = 0;

        do {
            if (*env == ',') {
		if (state == number || state == junk)
                    msg_level[i] = value;
                value = 0;
                state = fresh;
                if (++i >= MSG_CATEGORIES)
                    break;
            } else if (state != junk && isdigit(*env)) {
                value *= 10;
                value += *env - '0';
                if (value > MSG_MAX_SEVERITY + 1) {
                    value = MSG_MAX_SEVERITY + 1;
                    state = junk;
                } else
                    state = number;
	    } else if (*env == '-') {
		value = MSG_MAX_SEVERITY + 1;
		state = junk;
	    } else if (*env == '+') {
		value = 0;
		state = junk;
            } else if (state == number) /* non-digit, change state */
                state = junk; /* so "4k5, 7" becomes "4,7", not "45,7" */
        } while (*++env);

        if (state == number || state == junk)
            msg_level[i] = value;
    }
#endif
}


void
msg_printf(const char *fmt, ...)
{
    char    buff[1024];
    va_list args;
    register short int     prelen, msglen;

    if (!msg_init_done)
	msg_init();

    if (msg_parse_preface[msg_category]) {
        struct tm *tm;
	time_t     clock;
	char      *fmt;

        fmt = parse_preface(msg_preface[msg_category]);
        time(&clock);
        tm = localtime(&clock);
        prelen = strftime(buff, sizeof(buff), fmt, tm);
        if (prelen == 0)		/* Error condition from strftime */
            msg_fatal("Buffer too small");
        free(fmt);
    } else {
        strncpy(buff, msg_preface[msg_category], sizeof(buff));
	prelen = strlen(msg_preface[msg_category]);
    }

    va_start(args, fmt);
#if defined(_POSIX_SOURCE)
    msglen = vsprintf(buff + prelen, fmt, args);

    if (prelen + msglen > sizeof(buff))
        msg_fatal("Buffer too small");
#else
    /* BSD vsprintf() returns the first argument - can't be used
     * for detecting buffer over-runs.
     */
    (void) vsprintf(buff + prelen, fmt, args);
#endif

    va_end(args);
    fprintf(msg_file, "%s\n", buff);
}


static char *
parse_preface(const char *template) {
    char                    *fmt;
    size_t                   fmt_size;
    ptrdiff_t                p = 0;
    enum { normal, percent } state = normal;

    fmt_size = strlen(template) + 1;
    fmt = (char *) malloc(fmt_size);
    if (fmt == NULL)
        msg_fatal("Out of memory");

    for (; *template; template++) {
        if (p > fmt_size) {
            fmt_size += 16;
            fmt = realloc(fmt, fmt_size);
            if (fmt == NULL)
                msg_fatal("Out of memory");
        }
        fmt[p++] = *template;
        if (state == percent) {
            if (*template == 's')
#if defined(_POSIX_SOURCE)
                p += sprintf(fmt + p - 2, "%d", msg_severity) - 2;
#else
                p = strchr(sprintf(fmt + p - 2, "%d", msg_severity), 0) - fmt;
#endif
            state = normal;
        } else if (*template == '%')
            state = percent;
    }
    fmt[p] = 0;
    return fmt;
}

static void
msg_fatal(char *msg)
{
    fprintf(msg_file, "Fatal error: %s\n", msg);
    fclose(msg_file);
    exit(1);
}
