/*
 * $Header: /home/kline/devel/atom/RCS/atom.c,v 2.27 2009/05/26 19:20:04 kline Exp kline $
 * atom.c:  quote filter plain ASCII textfiles into TeX-formatted (default)
 * or HTML quotation marks.  Some miscellaneous markups as well.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include "macros.h"
#include "msg.h"
#include "atom.h"

extern char *version_string;  /* track */
extern char *optarg;

static int okay = 0, /* is is okay to return our modified string to main?  */
blockquotes     = 0, /* do we have any <BLOCKQUOTE></BLOCKQUOTE> tags  */
nblockquotes    = 0, /* if (blockquotes), how many? */
qcnt,                /* keep count of the quotes handled per line/buffer   */
qeven = 0,	     /*  is quote count even? if not, = 1 is an err */
html =  1,           /* hypertext markup language */
latex = 0,           /* LaTeX: */
nlcount = 0,	     /* newline count:: for asterisk line-err count      */
do_markup = 1,       /* Do markup, unless a CAP flag indicates otherwise */
quote_only= 0,       /* Perform quote-transformations only: leave file as-is */
italics_only=0,      /* Perform italics-only filtering                   */
no_quotes=0,	     /* leave " and " alone */
wquo=0,              /* switch to ask if reader wants to be asked about quotes */
Nquotes;

static short int spaceindented_paragraph = 0; /* for future xlation to TABs */

int
main(int argc, char **argv)
{
   char *program = argv[0];
   int e_status;
   void leave();
   void usage();

   signal (SIGINT, leave); signal (SIGQUIT, leave);

   msg_init();  /* pull in Kjetil Homme's debug library */
   MSG(TRACE, 1, ("In main(), argc = (%d), *argv = [%s]", argc, *argv));
   if (argc < 2)
   {
      usage(program);
      exit (1);
   }
   e_status = qfilter(argc, argv, program);
   exit(e_status);
}

static int
qfilter(int argc, char **argv, char *program)
{
   void putheader (int, char *, int , char *, FILE *);
   void puttailing (int, int, FILE *);
   extern int optind;
   int getopt(int argc, char * argv[], char *optstring);
   register FILE *fp_in, *fp_out;
   //char *a_err="\tNumber must be non-zero, postive, and *reasonable*: < 6";

   char buf[1024], *infilename, *out_buf, outputname[128];
   char *scan_buf_0(char *);
   char *scan_buf_1(char *);
   register short c;
   unsigned short num = 0; /* arbitrary given number of paragraph-space indents
			    * specified by the user.
			    */
   short noquotes_and_centered = 0;

   /* switches */
   short int centering = 0, /* handle space-centering in ASCII: rare     */
     ellipse = 0,           /* default is to NOT handle ellipses (TeX)   */
     italics = 1,           /* default is to handle italics              */
     pspace  = 0,           /* default: not add physical-space break     */
     squeeze = 0;           /* by default, do not remove excess newlines */

   int no_spaceindicated_paragraphs= 0;/* is a PP  indicated by 8 spaces */

   int TabdentedText(char *s);
   void usage();

   while (( c = getopt (argc, argv, "0123456789B:ceIihpQstvw?")) != EOF)
   {
      switch(c)
      {

      case '0': case '1': case '2': case '3': case '4': case '5': case '6':
      case '7': case '8': case '9':
	num = (10 * num + c - '0');
	 no_spaceindicated_paragraphs = 0;/* allow space-indenting       */
	 break;
      case 'B':
        if ((nblockquotes = (unsigned int)atoi(optarg)) == 0)
        {
          fprintf(stderr, "\nError: -a %s::  %s\n\n", *argv, optarg);
        }
	if (nblockquotes <= 0 || nblockquotes >= 6)
	{
	  printf("Only 1 to 5 \"<BLOCKQUOTES>\" allowed,  You entered (%d)\n", nblockquotes);
          usage(program);
	}


      case 'c':
         centering = 1;                    /* we want centering          */
	 no_spaceindicated_paragraphs = 1; /* disallow 8 spaces == PP    */
         break;
      case 'e':
         ellipse = 0;
         break;
      case 'I':
         italics_only = 1;
         break;
      case 'i':
         italics = 0;
         break;
         //html = 1;
         //latex = ellipse = 0; /* don't touch ellipse in HTML */
         //break;
      case 'l':
      case 't':
         html = 0;
         latex = 1;
         break;
      case 'p':  /* add <BR> tags or \vspace*{.5in} for physical space breaks */
	 pspace = 1;
	 if (squeeze)
	 {
	    fprintf(stderr, SQUEEZE_PHYSICAL_ERR);
	    exit(1);
	 }
	 break;
      case 'Q':  /* Do quote-transformation ONLY                */
	quote_only = 1;
	break;
      case 's':  /* squeeze excess newlines from processed file */
	 squeeze = 1;
	 if (pspace)
	 {
	    fprintf(stderr, SQUEEZE_PHYSICAL_ERR);
	    exit(1);
	 }
	 break;
      case 'v':  /* print version_string to stdout             */
         printf ("atom %s", version_string);   /* use _progname */
	 exit(0);
	 break;
      case 'w':  /* Does user want matching-quotes info?                */
	wquo = 1;
	break;
      case 'h':
      case '?':
      default:
         usage(program);
         break;
      }
   }
   argc -= optind;
   argv += optind;

#ifdef DEFAULTHTML
   if (!latex && !html && !quote_only)
   {
	fprintf(stderr, "You must specify either TeX or HTML output.\n");
	usage (program);
	exit(1);
   }
#endif // DEFAULTHTML
   while (argc > 0)
   {
      nlcount = 0;
      if ((fp_in = fopen (*argv, "r")) == NULL)
      {
         (void) fprintf(stderr, "Can't open file <%s>\n", *argv);
	 return (FATAL);
      }

      infilename = *argv;
      clear(outputname, 128);
      if (quote_only == 1)
      {
	 /*
	  * if quote_only option, name outputfile <file>.QO
	  */
	 (void)sprintf(outputname, "%s.QO", infilename);
      }
      else if (italics_only == 1)
      {
	 /*
	  * if italics_only option, name outputfile <file>.QO
	  */
	 (void)sprintf(outputname, "%s.IO", infilename);
      }
      else
      {
        if (latex)
	 (void)sprintf(outputname, "%s.tex", infilename);
        else if (html)
	 (void)sprintf(outputname, "%s.html", infilename);
      }

      if ((fp_out = fopen (outputname, "w")) == NULL)
      {
         (void) fprintf(stderr, "Can't open file_out \n");
	 return (FATAL);
      }

      if (do_markup)
      {
	if (quote_only == 0 && italics_only == 0)
	{
           putheader (html, infilename, latex, outputname, fp_out);
	}
      }

      /*
       * begin reading the input file here
       */
      while (fgets(buf, sizeof(buf), fp_in))
      {
	static unsigned int sqcount = 0;
         /* 
          *  A leading newline always indicates a paragraph at the
          *  following line that does not lead with a newline.
          *  True both for TeX and HTML.
	  *
	  *  Add a second case where the line begins with an initial
	  *  tab and we also have a paragraph. Or eight or more leading
	  *  spaces.
          */
	nlcount++;
	//printf("DEBUG: nlcount = (%d)\n", nlcount);
	if (squeeze)
	{

                if (*buf == '\n' && sqcount < ONE_NEWLINE  )
                {
                        sqcount++;
                }
                if (*buf == '\n' && sqcount > 0)
                {
                        continue;
                }
                else if (*buf != '\n')
                {
			/**continue; */
                }
                if (sqcount == ONE_NEWLINE  )
                {
                        sqcount = 0;
                        continue;
                }
	}
	if (pspace)
	{
		/*
		 * what i want to do here is print a break if there are
		 * 5 consec newlines and not print [HT]SPACE if there
		 * > 5 or < 5 consec newlines.
	 	 */
		if (H_TABEQUALS(buf))   /* === undocumented: my own use === */
		{
				put_break(html, latex, fp_out);
				continue;
		}
	        Do_Physical_space(buf, html, latex, fp_out);
	}

         Nquotes = GetNumberOfQuotes(buf);
	 if (odd(Nquotes))
	 {
		if (wquo)
		{
		printf("odd number of quotes on line #%d, source file [%s].\n", 
			nlcount, infilename);
		fprintf(stdout, "%s", buf);
		if (default_yn("Continue? [Y/n]?", 'y'))
		{  continue; }
		else {exit(0);}
		}
	 }
	 if (centering)
	 {
            unsigned int nleadingspaces = NumberOfLeadingSpaces(buf);
	    char *centered = StepPast(buf, nleadingspaces);
            unsigned int cslength = CenterStringLength(centered);
            if (DetermineCenteredness(nleadingspaces, cslength) )
	    {
		if (Nquotes == 0)
			noquotes_and_centered = 1;
		(void) DoCenterOutput(buf, fp_out, html, latex);
                MSG(TRACE, 3, ("\n%s\nis centered\n", buf));
	    }
	 }

	 if (!no_spaceindicated_paragraphs)  /* normal PP */
	 {
            if (NewlineOrTab(buf[0]) || IsEmbeddedTab(buf)|| 
		LeadingSpaces(buf, num))
            {
               MSG(TRACE,1,("No centering: Paragraph ahead: buf = [%s]\n",buf));
	       /*
               printf("DEBUG: No centering: Paragraph ahead: buf = [%s]\n",buf);
		*/

            }
            else
            {
               MSG(TRACE,1,("No centering: No paragraph ahead: buf = [%s]\n",
		  buf));
            }
	 }
	 else if (no_spaceindicated_paragraphs)
	 {
            if (NewlineOrTab(buf[0]) || IsEmbeddedTab(buf) )
            {
               MSG(TRACE,1,("CENTERING: Paragraph ahead: buf = [%s]\n", buf));
            }
            else
            {
               MSG(TRACE,1,("CENTERING: Not paragraph ahead: buf = [%s]\n", 
		  buf));
            }
	 }
#ifdef NOT_WORK
#endif 


         /*
          * default is to handle ellipse properly
          */
	 if (quote_only == 0 && italics_only == 0)
	 {
           if (ellipse)
           {
              (void)InsertSubstring(buf, ELIPSE, LDOTS, strlen(buf)+SLUSH);
           }
	 }

         /*
          * lines without quotes, including newlines,
          * leading or otherwise
          */
         if (Nquotes == 0)
         {
	    if (noquotes_and_centered) 
	    {
	       noquotes_and_centered = 0;
	       continue;
	    }
	    else
	    {
	       ProcessText (fp_out, buf, buf, nlcount, infilename);
               continue;
	    }
         }

         /*
          * last was a '' quote; eop.  or next _must_ be
          * `` quotes.
          */
	 if (italics_only == 1)
	 {
	    ProcessText (fp_out, buf, buf, nlcount , infilename);
	 }
	 else
	 {
         if (even(Nquotes) && !(IsNewParagraph(buf)))
         {
            out_buf = scan_buf_1(buf);
	    ProcessText (fp_out, buf, out_buf, nlcount , infilename);
            continue;
         }
         if (even(Nquotes) && (IsNewParagraph(buf)) ) 
         {
            out_buf = scan_buf_0(buf);
            MSG(TRACE,1,("EVEN && NewParagraph:  out_buf = [%s]\n", out_buf));
	    ProcessText (fp_out, buf, out_buf, nlcount, infilename);
            continue;
         }
         if (odd(Nquotes) && IsNewParagraph(buf) )
         {
            out_buf = scan_buf_0(buf);
            MSG(TRACE,1,("ODD && NewParagraph:  out_buf = [%s]\n", out_buf));
	    ProcessText (fp_out, buf, out_buf, nlcount, infilename);
            continue;
         }
         if (odd(Nquotes) && !(IsNewParagraph(buf)) )
         {
            out_buf = scan_buf_1(buf);
            MSG(TRACE,1,("ODD && NotNewParagraph:  out_buf = [%s]\n", out_buf));
	    ProcessText (fp_out, buf, out_buf, nlcount , infilename);
            continue;
         }
	 }

      }

      /*
       * tailing tags or comment line for either markup language 
       */
      if (do_markup)
      {
	if (quote_only == 0 && italics_only == 0)
	{
           puttailing (html, latex, fp_out);
	}
      }


      (void)fclose (fp_out);
      (void)fclose (fp_in);
      ++argv, argc--;
   }
   return (NO_ERRS);
}


/*
 * Check for beginning ``Quote, with a letter, digit, or ellipse following
 * on that line.''   If ss, we've got an exception, and include a LQUOTE
 * rather than a closing RQUOTE
 */
static int
CheckForLeftException (char *s)
{
   extern int Nquotes;
   int exception = 0;

   if (Nquotes == 1)
   {
      if (BeginningQuote(s))
      {
         exception = 1;
      }
   }
   return exception;
}

/*
 * step past n bytes; probably should be a macro...
 */
char *
StepPast(char *s, unsigned int n)
{
   register unsigned int i;
   for (i = 0; i < n; i++) s++;
   return (s);
}



static char *
ProcessItalicizedSymbols(char *s)
{
   register unsigned short int byte_count, count =  0, i = 0;
   char *chopped, *lhs, ibuf[1024], *rhs, *sp = s;
   char *left(char *, unsigned int);
   static char rbuf[1024]; 

   MSG(TRACE, 4, ("Entering ProcessItalicizedSymbols: s = [%s]\n", s) );
   if (quote_only == 1)
   {
     return (s);
   }

   while (*s)
   {
      if (*s == I_DELIM)
         count++;
      s++;
   }
   if (count >= 2)
   {
      byte_count = length_to_symbol(sp, I_DELIM);
      rhs = chopped = StepPast(sp, byte_count+1);
      while (*chopped)
      {
         if (*chopped != I_DELIM)
            ibuf[i++] = *chopped++;
            else
            break;
      }
      ibuf[i] =  '\0';
      /*
             * here, get both the left-hand side (up _to_ the first '*')
             * and the right-hand side (after the second '*');  and paste
             * the new string together in rbuf[]
             */
      while (*rhs != I_DELIM)
         rhs++;
      rhs++;
      lhs = left(sp, byte_count);
      if (latex)
      {
         (void)sprintf(rbuf, "%s{%cem %s%c%c}%s", 
             lhs, BSLASH, ibuf, BSLASH, SLASH, rhs );
      }
      else if (html)
      {
         (void)sprintf(rbuf, "%s<EM>%s</EM>%s", lhs, ibuf, rhs );

      }

   MSG(TRACE, 4, ("ProcessItalicizedSymbols() returning rbuf = [%s]\n", rbuf));

      return (rbuf);
   }
   return (sp);
}
/*
static int
get_nquotes(char *s)
{
   return(get_nsymbols(s, DQUOTE));
}
*/

static int
get_nsymbols(char *s, char symbol)
{
   register int count =  0;
   while (*s)
      {
         if (*s == symbol)
            count++;
         s++;
      }
   return (count);
}

static int
get_number_isyms(char *s)
{
   return(get_nsymbols(s, I_DELIM));
}

static int
BeginningQuote (char *s)
{
   /* alnum, ellipse, apostrophe, interior [single] quotes, or ital delimitor */
   /* or "{" which indicate a leading {\ldots} markup */
   /* or a leading italicized word */

   if (isalnum(s[1]) || s[1] == '\'' || s[1] == '`' || s[1] == I_DELIM ||
       (s[1]== '.' && s[2] == '.' && s[3]== '.') || s[1] == '{'  )
   {
      return (1);
   }
   return(0);
}


static int
IsNewParagraph( char *buf)
{
   unsigned int byte_count, retval = 0;
   char  *chopped_to_quote;
   MSG(TRACE,1, ("entering IsNewParagraph: qcnt = (%d), buf = [%s]\n",  
       qcnt, buf));

   byte_count = length_to_symbol(buf, DQUOTE);
   chopped_to_quote = StepPast(buf, byte_count);
   if (BeginningQuote (chopped_to_quote) )
   {
      retval = 1;
   }
   else
      retval = 0;
   MSG(TRACE,1, ("ISNEW: returning (%d)\n", retval));
   return (retval);

}


static char *
left(register char *source, register unsigned int count)
{
   static char st[4096]; /* Increase this as required */

   (void)strncpy(st, source, count);
   st[count] = 0; /* Shorten it. */
   return (st); 
}

static int
length_to_symbol(register char *s, char symbol)
{
   register int len = 0;
   while (*s)
   {
      if (*s != symbol) s++, len++;
      else break;
   }
   return (len);
}

char *
putleftquote(char *save, char *buf, unsigned int count)
{
   static char rbuf[1024]; 

   (void)sprintf(rbuf, "%s%s%s", save, L_DQ, StepPast(buf, count+1) );
   return (rbuf);
}

static char *
putrightquote(char *save, char *buf, unsigned int count)
{
   static char rbuf[1024];
   MSG(TRACE,2, ("putrightquote: save = [%s], buf = [%s], StepPast = [%s]\n",
       save, buf, StepPast(buf, count)));

   if (CheckForLeftException (StepPast(buf, count)) )
   {
      (void)sprintf(rbuf, "%s%s%s", save, L_DQ, StepPast(buf, count+1));
      return (rbuf);
   }
   else
   {
      (void)sprintf(rbuf, "%s%s%s", save, R_DQ, StepPast(buf, count+1));
      return (rbuf);
   }
}


/* 
 * report italics-symbol errors on newlinecount N.
 */
static void
report_isym_err(short int nsymbols, char *buf, int nlc, char *file)
{
   (void)fprintf(stderr, "\n\tFile [%s]: Italics ERR seen on line (%d):\n%s", 
      file, nlc, buf);
   (void)fprintf(stderr, "Odd number, %d, of italics delimitors, '%c'\n\n", 
       nsymbols, I_DELIM);
}
char *
scan_buf_0 (char *buf)
{
   char *left(char *, unsigned int), save[1024], *savebuf;
   register unsigned int byte_count; /***,  i = 0; */
   static char *r_rbuf;  /* recursive return buffer */
   char *putleftquote(char *, char *, unsigned int);
   char *scan_buf_1(char *);
   static char dead_save[1024];

   MSG(TRACE,1,("entering scan_buf_0:  qcnt = (%d), buf = [%s]\n", qcnt, buf));

   clear(save, 1024);
   byte_count = length_to_symbol(buf, DQUOTE);
   savebuf = left(buf, byte_count);
   (void)strcpy(save, savebuf);
   r_rbuf = putleftquote(save,buf, byte_count);
   (void)strcpy(dead_save, r_rbuf);
   qcnt = GetNumberOfQuotes(r_rbuf);
   MSG(TRACE,2,("r_rbuf = [%s], qcnt = (%d)\n", r_rbuf,qcnt));

   if ( qcnt == 0)
   {
      okay = 1;
      return (dead_save);
   }
   else
   {
      MSG(TRACE,3,("calling scan_buf_1 with [%s], [%s]\n", r_rbuf, dead_save));

      r_rbuf = scan_buf_1(dead_save);
      if (okay == 1)
      {
         MSG(TRACE, 3, ("scan_buf_0::  returning [%s]\n", r_rbuf));
         return (r_rbuf);
      }
   }
}
char *
scan_buf_1 (char *buf)
{
   char *left(char *, unsigned int), save[1024], *savebuf;
   register unsigned int byte_count; /***,  i = 0;  */
   static char *r_rbuf;  /* recursive return buffer */
   char *putrightquote(char *, char *, unsigned int);
   char *scan_buf_0();
   static char dead_save[1024];

   MSG(TRACE,2,("entering scan_buf_1: qcnt = (%d),  buf = [%s]\n", qcnt, buf));

   clear (save, 1024);
   byte_count = length_to_symbol(buf, DQUOTE);
   savebuf = left(buf, byte_count);
   (void)strcpy(save, savebuf);
   r_rbuf = putrightquote(save,buf, byte_count);
   (void)strcpy(dead_save, r_rbuf);
   qcnt = GetNumberOfQuotes(r_rbuf);
   MSG(TRACE,3, ("Rr_rbuf = [%s], qcnt = (%d)\n", r_rbuf, qcnt));

   if (qcnt == 0)
   {
      MSG(TRACE,3,("scan_buf_1: qcnt = 0, returning to main = [%s]\n", 
          dead_save));
      okay = 1;
      return (dead_save);
   }
   else
   {
      MSG(TRACE,3,("calling scan_buf_0 with [%s], [%s]\n",r_rbuf,dead_save));
      r_rbuf =  scan_buf_0(dead_save);
      if (okay == 1)
      {
         MSG(TRACE,3,("scan_buf_1: qcnt = (%d), returning to main = [%s]\n", 
             qcnt, r_rbuf));
         return (r_rbuf);
      }
   }
   return 0;
}

#include <stddef.h>
#include <stdlib.h>
static char *
strsave(const char *str)
{
   register unsigned short int len;
   register char *copy;

   len = strlen(str) + 1;
   if (!(copy = malloc((int)len)))
      return((char *)NULL);
   bcopy(str, copy, len);
   return(copy);
}

static void
usage(char *prog)
{
   (void)fprintf(stderr,
     "\tUsage %s: [-t] [-B {1 to 5}][-eGipQ[0-9]s] file...\n", prog);
   (void)fprintf(stderr, "\tRead the man page for tutorial-like coverage.\n");
}



/*
 * major processing of each line done here
 */
void
ProcessText(FILE *fp_out, char *buf, char *out_buf, int nlc, char *file)
{
   register short  j,  number_italics_symbols = 0;
   static int paragraphLineCount = 0;
   char *nbuf = NULL, *ProcessItalicizedSymbols();
   char *strsave();

   MSG(TRACE, 4, ("buf = [%s]\nout_buf = [%s]\n", buf, out_buf) );


	    /*
	     * handle italicization with or without paragraphing.
	     */
            number_italics_symbols = get_number_isyms(buf);
            if (odd(number_italics_symbols))
            {
	       if (quote_only == 0)
	       {
                  report_isym_err(number_italics_symbols, buf, nlc, file);
	       }
            }
            else if (number_italics_symbols == 2) /* most often */
            {
               nbuf = ProcessItalicizedSymbols(out_buf); 
		  if (quote_only == 0 && italics_only == 0)
		  {
                     if (html)
                     {
	        /* Paragraph TAG for hypertex markup */
                if (buf[0] == '\n' )
                {
			//puts("HERE-1");
                        if ( paragraphLineCount == 0)
                        {
                                paragraphLineCount++;
                                //fprintf(stdout, "<P>");
                        	(void)fprintf(fp_out, "<P>");
                        }
                }
                else if (paragraphLineCount == 1)
                {
                        paragraphLineCount = 0;
                }

                     }
                     else if (latex)
                     {
		        /* In LaTeX paragraphs are denoted by newline breaks */
                        (void)fprintf(fp_out, "\n");
                     }
		  }
               (void)fprintf(fp_out, "%s", nbuf);
            }
            else if (number_italics_symbols > 2)
            {
               nbuf = (char *)strsave(out_buf);
               for (j = 0; j < (number_italics_symbols/2); j++)
               {
                  nbuf = ProcessItalicizedSymbols(nbuf);
                  nbuf = (char *)strsave(nbuf);
               }
		  if (quote_only == 0 && italics_only == 0)
		  {
                     if (html)
                     {
                if (buf[0] == '\n' )
                {
			//puts("HERE-2");
                        if ( paragraphLineCount == 0)
                        {
                                paragraphLineCount++;
                        	(void)fprintf(fp_out, "<P>");
                                //fprintf(stdout, "<P>");
                        }
                }
                else if (paragraphLineCount == 1)
                {
                        paragraphLineCount = 0;
                }

                     }
                     else if (latex)
                     {
		        /* In LaTeX paragraphs are denoted by newline breaks */
                        (void)fprintf(fp_out, "\n");
                     }
		  }
               (void)fprintf(fp_out, "%s", nbuf);

	       if (nbuf)
	       {
		free(nbuf);
	       }
            }
            if (number_italics_symbols == 0)
            {
		  if (quote_only == 0 && italics_only == 0)
		  {
                     if (html)
                     {
                if (buf[0] == '\n' )
                {
			//puts("HERE-3");
                        if ( paragraphLineCount == 0)
                        {
                                paragraphLineCount++;
                                //fprintf(stdout, "<P>");
                        (void)fprintf(fp_out, "<P>\n");
                        }
                }
                else if (paragraphLineCount == 1)
                {
                        paragraphLineCount = 0;
                }

                     }
                     else if (latex)
                     {
		        /* In LaTeX paragraphs are denoted by newline breaks */
                        (void)fprintf(fp_out, "\n");
                     }
		  }
               (void)fprintf(fp_out, "%s", out_buf);
            }
	/**printf("nbuf = (0%o)\n", nbuf);**/

}
/*
 * If we have a line with leading spaces followed by a tab, we've got a
 * paragraph
 */
static int
IsEmbeddedTab(char *s)
{
	register unsigned int paragraph = 0;

	while (*s)
	{
		if (*s == ' ')
		{
			s++;
		}
		if (*s == '\t')
		{
			paragraph = 1;
			break;
		}
		s++;
	}
	return (paragraph);
}

static int
LeadingSpaces(char *s, int number_specified)
{
	extern short int  spaceindented_paragraph;
	register unsigned int space_count = 0, paragraph = 0;
	while (*s)
	{
		if (isalnum(*s) )
			break;
		if (number_specified == 0) /* none, or not specified */
		{
			if (*s == ' ')
				space_count++;
			/*
			 * choose explicity a 5- or 8-space 
			 * indented line as a
			 * paragraph.  Any other number of 
			 * leading-space indentations
			 * do not indicate a paragraph.
			 */
			switch (space_count)
			{
				case 1:
				case 2:
				case 3:
				case 4:
					paragraph = 0;
					break;
				case 5:
					paragraph = 1;
					spaceindented_paragraph = 1;
					break;
				case 6:
				case 7:
					paragraph = 0;
					break;
				case 8:
					paragraph = 1;
					spaceindented_paragraph = 1;
					break;
				default:
					paragraph = 0;
					break;
			}
		}
		else
		{
			if (*s == ' ')
				space_count++;
			if (space_count == number_specified)
				paragraph = 1;
			else 
				paragraph = 0;
		}
		s++;
	}
	/**printf("LS: buf = %s, paragraph = (%d)\n", s, paragraph); **/
	return (paragraph);
}

void
putheader (int html, char *ifn, int latex, char *ofn, FILE *fp)
{
      if (html)
      {
            /* header HTML tag  */
            (void)fprintf(fp, HHEADER );
            //(void)fprintf(fp, TBEGIN );
            (void)fprintf(fp, "<! @(#) %s translated from %s !>\n", ofn, ifn );
            (void)fprintf(fp, "<HTML>\n" );
            (void)fprintf(fp, HBEGIN );
	    switch(nblockquotes)
	    {
		case 1:
		    (void)fprintf(fp, BBEGIN1);
		    break;
		case 2:
		    (void)fprintf(fp, BBEGIN2);
		    break;
		case 3:
		    (void)fprintf(fp, BBEGIN3);
		    break;
		case 5:
		    (void)fprintf(fp, BBEGIN5);
		    break;
		case 4:
		default:
		    (void)fprintf(fp, BBEGIN4);
		    break;
	    }
      }
      else if (latex)
      {
            /* concluding comments for TeX-style docs  */
            (void)fputs(THEADER, fp );
            (void)fprintf(fp, " <<%s>> translated from <<%s>>", ofn, ifn ); 
	    (void)fputs(TCOMMENT, fp ); (void)fputs("\n", fp );
            (void)fputs(TCOMMENT  , fp ); (void)fprintf(fp, TATTRIBUTE );
      }

}

void
puttailing (int html, int latex, FILE *fp)
{
      if (html)
      {
            /* general HTML header lines and tags  */
	    switch(nblockquotes)
	    {
		case 1:
		    (void)fprintf(fp, BEND1);
		    break;
		case 2:
		    (void)fprintf(fp, BEND2);
		    break;
		case 3:
		    (void)fprintf(fp, BEND3);
		    break;
		case 5:
		    (void)fprintf(fp, BEND5);
		    break;
		case 4:
		default:
		    (void)fprintf(fp, BEND4);
		    break;
	    }
            (void)fprintf(fp, HEND );
            (void)fprintf(fp, HTAIL );
      }
      else if (latex)
      {
            /* header comments for TeX-style docs  */
            (void)fputs(TTAIL, fp );
      }
}

#ifdef WWW
lookahead (FILE *fp)
{
        char  lookaheadc;               /* lookahead character  */
	short int rval = 0;

	if (lookaheadc = getc (fp))
        {
                if (lookaheadc != '\n') 
                {
                        rval = 1;
                }
                else
                {
                        rval = 0;
                }
        }
	ungetc(lookaheadc, fp);
	return rval;
}
put_break(int html, int latex, FILE *fp)
{

	if (html)
	{
		fprintf(fp, HSPACE);
	}
	else if (latex)
	{
		fprintf(fp, TSPACE);
	}
}
#endif

/*
 *  Does the ascii file have tab-indented blocks of text?  perhaps with
 *  /n/t/t/t embedded tabs?  Using ``atom -h -T <file>''   understands
 *  tab indentation.
 */

int
TabdentedText(char *s)
{
  static int nnls = 0;
  char *tabs;
  int returnval = 0;

//puts("DEB: enteing Tab");

  if (s[0] == '\n')  // newline
  {
    returnval = 1;
  }
  else if (s[0] == '\t')  // tab or tabs are considered to be stuff (as TEXT)
  {
     if ((tabs = (char *)strchr(s, '\t')) )
     {
	returnval = 0;
	while (*s)
	{
	  if (isalnum(*s) || isprint(*s))
	  /*if (isalnum(*s))*/
	  {
	    returnval = 0;
	  }
	  s++;
	}
     }
  }
  else returnval = 0;

  /*
   * here we step past multiple newlines to avoid cluttering the output
   * file with multiple <P> tags.
   */
  if (returnval)
  {
    nnls++;
  }
  if (nnls > 1)
  {
    nnls =  returnval = 0;
    return 0;
  }
  if (returnval==1) 
  {
    /*puts("ONE"); */
    return 1;
  }
  else
  return 0;

}
